<?php
use Livewire\Volt\Component;
use Illuminate\Support\Facades\Auth;
use App\Models\Z_A_BOTS_SYSTEMS\Event4TApp;
use App\Models\SRO_VT_SHARD\Char;
use App\Models\SRO_VT_PROXY\_Players;
use Carbon\Carbon;

new class extends Component {
    public $currentCharacterName = null;
    public $currentCharacterAvatar = null;
    public $userOwnsCharacter = false;
    public $remainingTimeInSeconds = 0;
    public $availableCharacters = [];
    public $userCharCount = 0;
    public $selectedCharId = null;
    public $isLoading = false;
    public $showSelectionModal = false;
    public $daysSinceStart = 0;
    public $eventStartDate = null;
    public $messages = [];
    public $eventStatus = 'none'; // none, waiting, active, expired
    public $isOwner = false;
    public $eventData = null;

    public function mount()
    {
        $this->loadUserData();
    }

    public function loadUserData()
    {
        try {
            $user = Auth::user();

            if (!$user) {
                $this->addMessage('error', 'User not authenticated');
                return;
            }

            $username = $user->StrUserID ?? $user->username ?? $user->name;
            $userJID = $user->JID ?? $user->id;

            // Get user's characters IDs
            $charIDs = \App\Models\SRO_VT_SHARD\_User::where('UserJID', $userJID)->pluck('CharID');

            // Check if user has any event records
            $eventRecord = Event4TApp::whereIn('CharID', $charIDs)->first();

            if (!$eventRecord) {
                // Scenario 1: No event record - show character selection
                $this->eventStatus = 'none';
                $this->userCharCount = $charIDs->count();
                if ($this->userCharCount > 0) {
                    $this->availableCharacters = Event4TApp::getAvailableChars($userJID);
                }
            } else {
                // Get current event data
                $this->eventData = $eventRecord;
                $this->currentCharacterName = $eventRecord->Charname;
                $this->currentCharacterAvatar = Char::getCharAvatarGifByCharname($eventRecord->Charname);

                // Check if user is the owner (same Hwid)
                $userSerial = \App\Models\SRO_VT_PROXY\_Players::where('StrUserID', $username)->value('Serial');
                $this->isOwner = ($userSerial && $userSerial == $eventRecord->Hwid);

                $startTime = Carbon::parse($eventRecord->Start);
                $now = Carbon::now();

                if ($startTime->isFuture()) {
                    // Scenario 2a: Event not started yet - show countdown to start
                    $this->eventStatus = 'waiting';
                    $this->remainingTimeInSeconds = $now->diffInSeconds($startTime, false);
                } else {
                    // Scenario 2b: Event is active - show main timer
                    $this->eventStatus = 'active';
                    $endTime = $startTime->addHours(18); // Assuming 18 hours duration

                    if ($endTime->isFuture()) {
                        $this->remainingTimeInSeconds = $now->diffInSeconds($endTime, false);
                    } else {
                        $this->eventStatus = 'expired';
                        $this->remainingTimeInSeconds = 0;
                    }
                }

                // If user is owner, show additional options
                if ($this->isOwner) {
                    $this->userOwnsCharacter = true;
                    $this->userCharCount = Event4TApp::countCharsForUser($userJID);
                    if ($this->userCharCount > 1) {
                        $this->availableCharacters = Event4TApp::getAvailableChars($userJID);
                    }
                }

                // Calculate days since registration
                $this->calculateDaysSinceStart($userJID);
            }

        } catch (\Exception $e) {
            \Log::error('4T-App loadUserData error:', [
                'message' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);
            $this->addMessage('error', 'Error loading user data: ' . $e->getMessage());
        }
    }

    public function calculateDaysSinceStart($userJID)
    {
        // Get the Start date for user's character
        $charIDs = \App\Models\SRO_VT_SHARD\_User::where('UserJID', $userJID)->pluck('CharID');

        if ($charIDs->isNotEmpty()) {
            $event = Event4TApp::whereIn('CharID', $charIDs)->first();
            if ($event && $event->Start) {
                $this->eventStartDate = Carbon::parse($event->Start)->format('Y-m-d H:i:s');

                // حساب عدد الأيام منذ تاريخ التسجيل حتى الآن
                $registrationDate = Carbon::parse($event->Start);
                $now = Carbon::now();

                // استخدام diffInDays مع false للحصول على عدد الأيام الصحيح
                $this->daysSinceStart = $now->diffInDays($registrationDate, false);

                // التأكد من أن القيمة موجبة دائماً وصحيحة
                $this->daysSinceStart = abs($this->daysSinceStart);
            }
        }
    }

    public function selectCharacterForEvent()
    {
        if (!$this->selectedCharId) {
            $this->addMessage('error', 'Please select a character');
            return;
        }

        $this->isLoading = true;

        try {
            // Get character details
            $char = Char::find($this->selectedCharId);

            if (!$char) {
                $this->addMessage('error', 'Character not found');
                $this->isLoading = false;
                return;
            }

            // Check if character is in-game
            $player = _Players::where('CharID', $char->CharID)->first();

            if (!$player || $player->cur_status == 0) {
                $this->addMessage('error', 'Character must be in-game to participate');
                $this->isLoading = false;
                return;
            }

            // Create new event record
            Event4TApp::create([
                'Hwid' => $player->Serial,
                'CharID' => $char->CharID,
                'Charname' => $char->CharName16,
                'Start' => Carbon::now()->addHours(18),
            ]);

            $this->addMessage('success', 'Character registered for event successfully!');
            $this->showSelectionModal = false;
            $this->selectedCharId = null;
            $this->loadUserData();

        } catch (\Exception $e) {
            $this->addMessage('error', 'Error registering character: ' . $e->getMessage());
        }

        $this->isLoading = false;
    }

    public function refreshData()
    {
        $this->loadUserData();
        $this->addMessage('success', 'Data refreshed successfully');
    }

    private function addMessage($type, $message)
    {
        $this->messages[] = ['type' => $type, 'message' => $message];
    }

    public function clearMessages()
    {
        $this->messages = [];
    }

    public function formatTime($seconds)
    {
        if ($seconds <= 0) return 'Expired';

        $hours = floor($seconds / 3600);
        $minutes = floor(($seconds % 3600) / 60);
        $secs = $seconds % 60;

        return sprintf('%02d:%02d:%02d', $hours, $minutes, $secs);
    }

    public function formatDays($days)
    {
        if ($days == 0) return 'Started today';
        if ($days == 1) return '1 day ago';
        return $days . ' days ago';
    }
};
?>
<div>
    <link rel="stylesheet" href="{{ asset('assets/css/4t-app-simple.css') }}">

    <div class="t-app-container space-y-6" x-data="{
        timer: @entangle('remainingTimeInSeconds'),
        currentCharacterName: @entangle('currentCharacterName'),
        currentCharacterAvatar: @entangle('currentCharacterAvatar'),
        eventStatus: @entangle('eventStatus'),
        isOwner: @entangle('isOwner'),
        userCharCount: @entangle('userCharCount'),
        availableCharacters: @entangle('availableCharacters'),
        daysSinceStart: @entangle('daysSinceStart'),
        eventStartDate: @entangle('eventStartDate'),
        isLoading: @entangle('isLoading'),
        messages: @entangle('messages'),
        showSelectionModal: @entangle('showSelectionModal'),
        selectedCharId: @entangle('selectedCharId'),

        init() {
            this.startTimer();
        },

        startTimer() {
            setInterval(() => {
                if (this.timer > 0) {
                    this.timer--;
                }
            }, 1000);
        },

        formatTime(seconds) {
            if (seconds <= 0) return '00:00:00';
            const hours = Math.floor(seconds / 3600);
            const minutes = Math.floor((seconds % 3600) / 60);
            const secs = seconds % 60;
            return String(hours).padStart(2, '0') + ':' +
                   String(minutes).padStart(2, '0') + ':' +
                   String(secs).padStart(2, '0');
        },

        getHours() {
            return Math.floor(this.timer / 3600);
        },

        getMinutes() {
            return Math.floor((this.timer % 3600) / 60);
        },

        getSeconds() {
            return this.timer % 60;
        },

        getTimerTitle() {
            if (this.eventStatus === 'waiting') return 'Event Starts In';
            if (this.eventStatus === 'active') return this.timer > 0 ? '4T App Usage Timer' : '4T Mobile App Ready';
            return 'Event Timer';
        },

        getTimerClass() {
            if (this.eventStatus === 'waiting') return 'text-orange-400';
            if (this.eventStatus === 'active') return this.timer > 0 ? 'text-blue-400' : 'text-green-400';
            if (this.eventStatus === 'expired') return 'text-red-400';
            return 'text-gray-400';
        },

        getStatusText() {
            if (this.eventStatus === 'waiting') return 'Waiting';
            if (this.eventStatus === 'active') return this.timer > 0 ? 'Counting Down' : 'Ready';
            if (this.eventStatus === 'expired') return 'Expired';
            return 'Unknown';
        },

        getStatusTranslateKey() {
            if (this.eventStatus === 'waiting') return 'page.4t-app.waiting';
            if (this.eventStatus === 'active') return this.timer > 0 ? 'page.4t-app.countdownTimer' : 'page.4t-app.appReady';
            if (this.eventStatus === 'expired') return 'page.4t-app.expired';
            return '';
        }
    }" wire:ignore.self>

        <!-- Loading State -->
        <div x-show="isLoading" class="flex items-center justify-center py-12" x-transition>
            <div class="animate-spin rounded-full h-12 w-12 border-b-2 border-blue-600"></div>
            <p class="mt-2 text-sm text-gray-600 ml-3" data-translate="page.4t-app.loading">Loading...</p>
        </div>

        <!-- Messages -->
        <div x-show="messages && messages.length > 0" class="fixed top-4 right-4 z-50 space-y-2" x-transition>
            <template x-for="(message, index) in messages" :key="index">
                <div class="px-4 py-3 rounded-lg shadow-lg backdrop-blur-sm"
                     :class="message.type === 'success' ? 'bg-green-500 text-white' : 'bg-red-500 text-white'"
                     x-show="true" x-transition.duration.300ms>
                    <div class="flex items-center justify-between">
                        <span x-text="message.message"></span>
                        <button @click="$wire.clearMessages()" class="ml-3 text-white/80 hover:text-white">
                            <i class="fas fa-times"></i>
                        </button>
                    </div>
                </div>
            </template>
        </div>

        <!-- Page Header -->
        <div class="t-header" x-show="!isLoading" x-transition>
            <div class="t-header-icon">
                <i class="fas fa-mobile-alt"></i>
            </div>
            <h2 class="t-header-title" data-translate="page.4t-app.title">4T-App Event System</h2>
            <p class="text-gray-600 dark:text-gray-400 text-lg" data-translate="page.4t-app.description">
                Manage your character participation in the 4T mobile application event
            </p>
        </div>

        <!-- Main Content Area -->
        <div class="main-content" x-show="!isLoading" x-transition.delay.100ms>

            <!-- Scenario 1: No Event Record - Character Selection -->
            <template x-if="eventStatus === 'none'">
                <div class="selection-container" x-data="{ isAnimating: false }">
                    <div class="selection-header">
                        <div class="selection-icon">
                            <i class="fas fa-user-plus"></i>
                        </div>
                        <h3 class="text-2xl font-bold text-gray-900 mb-2" data-translate="page.4t-app.joinEvent">Join 4T Event</h3>
                        <p class="text-gray-600" data-translate="page.4t-app.selectCharacterToJoin">Select a character to participate in the 4T mobile application event</p>
                    </div>

                    <template x-if="userCharCount > 0">
                        <div class="character-grid">
                            <template x-for="(character, index) in availableCharacters" :key="character.CharID">
                                <div class="character-card"
                                     :class="selectedCharId == character.CharID ? 'selected' : ''"
                                     :style="`animation-delay: ${index * 0.1}s`"
                                     @click="selectedCharId = character.CharID; $nextTick(() => {
                                         if (selectedCharId == character.CharID) {
                                             $el.querySelector('.character-check').style.animation = 'checkBounce 0.5s cubic-bezier(0.68, -0.55, 0.265, 1.55)';
                                         }
                                     })">
                                    <div class="character-avatar">
                                        <img :src="'/assets/images/avatars/' + character.RefObjID + '.gif'"
                                             :alt="character.CharName16"
                                             onerror="this.src='/assets/images/avatars/default.gif'"
                                             loading="lazy">
                                    </div>
                                    <div class="character-info">
                                        <h4 class="character-name" x-text="character.CharName16"></h4>
                                        <p class="character-id">ID: <span x-text="character.CharID"></span></p>
                                    </div>
                                    <div class="character-check" x-show="selectedCharId == character.CharID" x-transition.scale.origin.top.right>
                                        <i class="fas fa-check-circle"></i>
                                    </div>
                                </div>
                            </template>
                        </div>
                    </template>

                    <template x-if="userCharCount === 0">
                        <div class="no-characters">
                            <i class="fas fa-exclamation-triangle text-4xl text-gray-400 mb-4"></i>
                            <p class="text-gray-500" data-translate="page.4t-app.noCharactersAvailable">No characters available</p>
                        </div>
                    </template>

                    <div class="selection-actions" x-show="userCharCount > 0">
                        <button @click="isAnimating = true; $wire.selectCharacterForEvent().finally(() => isAnimating = false)"
                                class="join-btn"
                                :disabled="!selectedCharId || isLoading || isAnimating"
                                :class="{ 'animate-pulse': isAnimating }">
                            <i class="fas fa-rocket" :class="{ 'animate-spin': isAnimating }"></i>
                            <span data-translate="page.4t-app.joinEvent" x-text="isAnimating ? 'Joining...' : 'Join Event'"></span>
                        </button>
                    </div>
                </div>
            </template>

            <!-- Scenario 2: Event Record Exists - Show Timer and Character Info -->
            <template x-if="eventStatus !== 'none'">
                <div class="event-container">

                    <!-- Event Timer Section (Only show if timer is active or waiting) -->
                    <template x-if="eventStatus === 'waiting' || (eventStatus === 'active' && timer > 0)">
                        <div class="event-timer-section">
                            <div class="timer-header">
                                <h3 class="timer-title" x-text="getTimerTitle()"></h3>
                                <div class="timer-status" :class="getTimerClass()">
                                    <i class="fas fa-circle pulse"></i>
                                    <span x-text="getStatusText()" :data-translate="getStatusTranslateKey()"></span>
                                </div>
                            </div>

                            <div class="timer-display-container">
                                <div class="timer-display" :class="getTimerClass()">
                                    <div class="timer-segments">
                                        <div class="timer-segment" x-show="getHours() > 0">
                                            <div class="timer-number hours-animation" x-text="String(getHours()).padStart(2, '0')"></div>
                                            <div class="timer-label" data-translate="page.4t-app.hoursRemaining">Hours</div>
                                        </div>
                                        <div class="timer-separator" x-show="getHours() > 0">:</div>
                                        <div class="timer-segment">
                                            <div class="timer-number minutes-animation" x-text="String(getMinutes()).padStart(2, '0')"></div>
                                            <div class="timer-label" data-translate="page.4t-app.minutesRemaining">Minutes</div>
                                        </div>
                                        <div class="timer-separator">:</div>
                                        <div class="timer-segment">
                                            <div class="timer-number seconds-animation" x-text="String(getSeconds()).padStart(2, '0')"></div>
                                            <div class="timer-label" data-translate="page.4t-app.secondsRemaining">Seconds</div>
                                        </div>
                                    </div>
                                </div>
                            </div>

                            <div class="timer-description-section">
                                <template x-if="eventStatus === 'waiting'">
                                    <div class="timer-description waiting">
                                        <div class="description-icon">
                                            <i class="fas fa-clock"></i>
                                        </div>
                                        <p data-translate="page.4t-app.eventWillStart">Your event will start automatically at the scheduled time</p>
                                        <div class="countdown-subtitle" data-translate="page.4t-app.getReady">Get Ready for Your Adventure</div>
                                    </div>
                                </template>

                                <template x-if="eventStatus === 'active' && timer > 0">
                                    <div class="timer-description active">
                                        <div class="description-icon">
                                            <i class="fas fa-mobile-alt"></i>
                                        </div>
                                        <p data-translate="page.4t-app.canUseAppIn">You can use the 4T mobile app in</p>
                                    </div>
                                </template>
                            </div>
                        </div>
                    </template>

                    <!-- Success Message (Show when timer reaches 0 for active events) -->
                    <template x-if="eventStatus === 'active' && timer <= 0">
                        <div class="event-success-section">
                            <div class="success-container">
                                <div class="success-icon">
                                    <i class="fas fa-check-circle"></i>
                                </div>
                                <h3 class="success-title" data-translate="page.4t-app.appReady">4T Mobile App Ready</h3>
                                <p class="success-message" data-translate="page.4t-app.useAppNow">You can now use the 4T mobile application in-game and on the website!</p>
                                <div class="success-badge">
                                    <i class="fas fa-mobile-alt"></i>
                                    <span data-translate="page.4t-app.active">Active</span>
                                </div>
                            </div>
                        </div>
                    </template>

                    <!-- Character Info Section -->
                    <div class="character-info-section">
                        <div class="current-character-card">
                            <div class="character-avatar-large">
                                <img :src="currentCharacterAvatar || '/assets/images/avatars/default.gif'"
                                     :alt="currentCharacterName"
                                     onerror="this.src='/assets/images/avatars/default.gif'">
                            </div>
                            <div class="character-details">
                                <h4 class="character-name-large" x-text="currentCharacterName"></h4>
                                <div class="character-status">
                                    <template x-if="isOwner">
                                        <span class="status-badge owner">
                                            <i class="fas fa-crown"></i>
                                            Your Character
                                        </span>
                                    </template>
                                    <template x-if="!isOwner">
                                        <span class="status-badge viewer">
                                            <i class="fas fa-eye"></i>
                                            Viewing Event
                                        </span>
                                    </template>
                                </div>
                            </div>
                        </div>
                    </div>

                    <!-- Owner Controls (Only for character owner) -->
                    <template x-if="isOwner && eventStatus !== 'expired'">
                        <div class="owner-controls">
                            <template x-if="userCharCount > 1">
                                <button @click="showSelectionModal = true" class="control-btn secondary">
                                    <i class="fas fa-exchange-alt"></i>
                                    <span data-translate="page.4t-app.switchCharacter">Switch Character</span>
                                </button>
                            </template>

                            <button @click="$wire.refreshData()" class="control-btn primary">
                                <i class="fas fa-sync-alt"></i>
                                <span data-translate="page.4t-app.refresh">Refresh</span>
                            </button>
                        </div>
                    </template>

                    <!-- Event Statistics (Only for owner) -->
                    <template x-if="isOwner && eventStatus !== 'none'">
                        <div class="event-stats">
                            <div class="stat-card">
                                <div class="stat-icon green">
                                    <i class="fas fa-calendar-check"></i>
                                </div>
                                <div class="stat-content">
                                    <span class="stat-label" data-translate="page.4t-app.registrationDate">Registration Date</span>
                                    <div class="stat-value">
                                        <div class="stat-date"
                                             x-text="eventStartDate ? new Date(eventStartDate).toLocaleDateString('en-US', {
                                                 year: 'numeric', month: 'short', day: 'numeric'
                                             }) : 'N/A'"></div>
                                        <div class="stat-time"
                                             x-text="eventStartDate ? new Date(eventStartDate).toLocaleTimeString('en-US', {
                                                 hour: '2-digit', minute: '2-digit', hour12: true
                                             }) : ''"></div>
                                    </div>
                                </div>
                            </div>

                            <div class="stat-card">
                                <div class="stat-icon blue">
                                    <i class="fas fa-hourglass-half"></i>
                                </div>
                                <div class="stat-content">
                                    <span class="stat-label" data-translate="page.4t-app.daysSinceRegistration">Days Since Registration</span>
                                    <div class="stat-value">
                                        <span class="stat-number" x-text="Math.floor(Math.abs(daysSinceStart))"></span>
                                        <span class="stat-unit">days</span>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </template>
                </div>
            </template>
        </div>

        <!-- Character Selection Modal -->
        <div x-show="showSelectionModal"
             class="fixed inset-0 bg-black/75 backdrop-blur-sm flex items-center justify-center p-4 z-50"
             x-transition:enter="transition ease-out duration-300"
             x-transition:enter-start="opacity-0"
             x-transition:enter-end="opacity-100"
             x-transition:leave="transition ease-in duration-200"
             x-transition:leave-start="opacity-100"
             x-transition:leave-end="opacity-0">

            <div class="bg-white rounded-2xl shadow-2xl w-full max-w-2xl max-h-[80vh] overflow-hidden"
                 x-transition:enter="transition ease-out duration-300 transform"
                 x-transition:enter-start="opacity-0 translate-y-4 scale-95"
                 x-transition:enter-end="opacity-100 translate-y-0 scale-100"
                 x-transition:leave="transition ease-in duration-200 transform"
                 x-transition:leave-start="opacity-100 translate-y-0 scale-100"
                 x-transition:leave-end="opacity-0 translate-y-4 scale-95">

                <div class="modal-header">
                    <h3 class="text-xl font-bold text-gray-900" data-translate="page.4t-app.switchCharacter">Switch Character</h3>
                    <button @click="showSelectionModal = false" class="text-gray-400 hover:text-gray-600 p-1">
                        <i class="fas fa-times text-xl"></i>
                    </button>
                </div>

                <div class="modal-body">
                    <p class="text-gray-600 mb-6" data-translate="page.4t-app.selectCharacterDescription">
                        Choose a different character for the 4T event. The selected character must be online.
                    </p>

                    <div class="modal-character-grid">
                        <template x-for="character in availableCharacters" :key="character.CharID">
                            <div class="modal-character-card"
                                 :class="selectedCharId == character.CharID ? 'selected' : ''"
                                 @click="selectedCharId = character.CharID">
                                <div class="modal-character-avatar">
                                    <img :src="'/assets/images/avatars/' + character.RefObjID + '.gif'"
                                         :alt="character.CharName16"
                                         onerror="this.src='/assets/images/avatars/default.gif'">
                                </div>
                                <div class="modal-character-info">
                                    <h4 class="modal-character-name" x-text="character.CharName16"></h4>
                                    <p class="modal-character-id">ID: <span x-text="character.CharID"></span></p>
                                </div>
                                <div class="modal-character-check" x-show="selectedCharId == character.CharID">
                                    <i class="fas fa-check-circle"></i>
                                </div>
                            </div>
                        </template>
                    </div>
                </div>

                <div class="modal-footer">
                    <button @click="showSelectionModal = false" class="control-btn secondary">
                        <span data-translate="page.4t-app.cancel">Cancel</span>
                    </button>
                    <button @click="$wire.selectCharacterForEvent()"
                            class="control-btn primary"
                            :disabled="!selectedCharId || isLoading">
                        <span data-translate="page.4t-app.confirm">Confirm Selection</span>
                    </button>
                </div>
            </div>
        </div>
    </div>
</div>
